<?php

namespace WP_Rocket\Addon\ImageOptimization;

use WP_Rocket\Admin\Options;
use WP_Rocket\Admin\Options_Data;

/**
 * Manager for Image optimization options
 */
class OptionsManager {
	/**
	 * WP Options API instance
	 *
	 * @var Options
	 */
	private $options_api;

	/**
	 * WP Rocket Options instance
	 *
	 * @var Options_Data
	 */
	private $options;

	/**
	 * Constructor
	 *
	 * @param Options      $options_api WP Options API instance.
	 * @param Options_Data $options WP Rocket Options instance.
	 */
	public function __construct( Options $options_api, Options_Data $options ) {
		$this->options_api = $options_api;
		$this->options     = $options;
	}

	/**
	 * Enable Image optimization option
	 *
	 * @param string $unique_id CDN URL.
	 *
	 * @return void
	 * @since 3.5
	 */
	public function enable( $unique_id ) {
		$this->options->set( 'awp_image_optimization', 1 );
		$this->options->set( 'awp_image_optimization_unique_id', $unique_id );
		$this->options->set( 'cache_webp', 1 );

		$this->options_api->set( 'settings', $this->options->get_options() );

		rocket_clean_domain();
	}

	/**
	 * Disable CDN option, remove CDN URL & user token, delete RocketCDN status transient
	 *
	 * @return void
	 * @since 3.5
	 */
	public function disable() {
		$this->options->set( 'awp_image_optimization', 0 );
		$this->options->set( 'awp_image_optimization_unique_id', '' );
		$this->options->set( 'cache_webp', 0 );

		$this->options_api->set( 'settings', $this->options->get_options() );

		rocket_clean_domain();
	}

	/**
	 * Get cdn status.
	 *
	 * @return int
	 */
	public function get_cdn() {
		return $this->options->get( 'cdn', 0 );
	}

	/**
	 * Get current image optimization status.
	 *
	 * @return bool
	 */
	public function is_image_optimization_enabled() {
		return (bool) $this->options->get( 'awp_image_optimization', 0 );
	}

	/**
	 * Get unique id.
	 *
	 * @return string
	 */
	public function get_unique_id() {
		$id = $this->options->get( 'awp_image_optimization_unique_id', '' );

		if ( defined( 'AWP_UNIQUE_ID' ) ) {
			$id = AWP_UNIQUE_ID;
		}

		return (string) $id;
	}
}
